#include "MainGame.h"



MainGame::MainGame(sf::RenderWindow* hwnd, Input* input) : Level(hwnd, input)
{
	reset();	//Call reset in the constructor to begin setting up the level
	//Makes sense to do this as a separate function rather than in the constructor as we don't need to keep track of, delete and create new MainGame objects
}


MainGame::~MainGame()
{
}

void MainGame::reset()
{
	//	//	//	//
	delete enemyManager;
	delete itemManager;
	delete moveablesManager;		//	Delete our old objects
	delete gameMap;
	//	//	//	//	

	//	//	//	//
	Level::enemyManager = new EnemyManager();
	Level::itemManager = new ItemManager();
	Level::moveablesManager = new MoveablesManager();		//	Create new objects for this run of the level
	Level::gameMap = new GameMap();
	//	//	//	//	

	gameMap->load("Assets/Sprites/Tile Maps/1-1.png", window, itemManager, enemyManager);	//Call load to generate the game map from the 1-1.png tilemap

	backgroundTexture.loadFromFile("Assets/Sprites/skyBackground.png");	//Load our background texture and set its size to the window
	backgroundSprite.setTexture(&backgroundTexture);
	backgroundSprite.setSize(sf::Vector2f(11637, 768));

	//	//	//	//
	setupTextObject(&marioText);
	marioText.setString("MARIO");
	marioText.setPosition(70, 30);

	setupTextObject(&scoreText);
	scoreText.setString("000300");
	scoreText.setPosition(70, 60);

	setupTextObject(&coinsText);
	coinsText.setString("x00");
	coinsText.setPosition(380, 60);

	setupTextObject(&worldText);
	worldText.setString("WORLD");
	worldText.setPosition(600, 30);

	setupTextObject(&levelText);
	levelText.setString("1-1");			//	Setup all out text objects, set the appropriate strings and positions for the text to render
	levelText.setPosition(625, 60);

	setupTextObject(&timeText);
	timeText.setString("TIME");
	timeText.setPosition(850, 30);

	setupTextObject(&timeLeftText);
	timeLeftText.setString("180");
	timeLeftText.setPosition(850, 60);

	setupTextObject(&pausedText);
	pausedText.setString("GAME PAUSED");
	pausedText.setPosition(512, 384);
	//	//	//	//

	coinBesideCount.setPosition(sf::Vector2f(340, 50));	//Set the position of the coin next to the coin amount

	levelView.reset(sf::FloatRect(0, 0, 1024, 768));
	levelView.setViewport(sf::FloatRect(0, 0, 1.0f, 1.0f));		//We setup our level view, setting its size to the window, a standard viewpoint, and set the centre of the sidescrolling to the middle of the window
	levelView.setCenter(sf::Vector2f(512, 384));

	Timer::setLevelTime(400);	//Call setLevelTime to set 400 seconds for the player to complete the level
	Timer::timerStopped = false;

	if (mario.isPoweredUp())	//If Mario is powered up
	{
		mario.powerDown();	//We power him down for the start of the next run
		Level::audioManager->stopAllSounds(); //We call this so we don't play the power down sound
	}

	mario.setFillColor(sf::Color(255, 255, 255, 255));	//Set Mario to be fully opaque
	marioTexture.loadFromFile("Assets/Sprites/MarioSprites.png");	//Load the Mario sprite sheet
	mario.setInput(userInput);	//Mario is controlled by out userInput object
	mario.setTexture(&marioTexture);	//Set mario's texture to the sprite sheet
	mario.setPosition(100, 600);	//Set Mario to the beginning of the level
	mario.setAlive(true);	//Mario is alive again as the level is beginning
	mario.setVelocity(0, 0);	//Kill any velocity Mario may have gained while spawning in
	mario.inputsAllowed = true;	//We're allowed to control Mario again
	mario.setWon(false);	//Mario has not won the level during reset

	hasBackgroundStarted = false;
}

void MainGame::update(float dt)
{
	scoreText.setString(updateScoreText());	//Set the score text string to the return value of updateScoreText
	coinsText.setString("x0" + std::to_string(mario.getCoins()));	//Set the coin text to "x0" plus the string return type of getCoins

	if (!mario.isAlive())	//If Mario has died
	{
		if (Timer::setWaitTime(4.0, dt))	//After 4 seconds of Mario playing his death animation
		{
			if (mario.getLives() <= 0)	//If Mario has run out of lives
			{
				window->setView(window->getDefaultView());
				currentLevel = Level::QueryLevel(LevelType_GameOver);	//Go to the game over screen
				reset();	//We call reset to set up the level for the next run when the player restarts
				mario.setLives(3);	//We set Mario's lives back to 3
				mario.removeCoins();						//	
				mario.removeScore(mario.getScore());		//		We remove all accumulated coins and score from Mario
				Level::audioManager->playSoundByName("GameOver");
			}

			else if (Timer::getLevelTimeLeft(dt) <= 0)	//If Mario has run out of time
			{
				window->setView(window->getDefaultView());
				currentLevel = Level::QueryLevel(LevelType_TimeUp);	//Go to the time up screen
				reset();	//We call reset to set up the level for the next run
			}

			else    //Mario has died via an enemy or falling into a pit... a non-special condition
			{
				window->setView(window->getDefaultView());
				currentLevel = Level::QueryLevel(LevelType_Lives);	//Go to the lives screen
				reset();	//We call reset to set up the level for the next run
			}
		}
	}

	else if (mario.getWon())	//If Mario has won
	{
		if (mario.walkToCastle())	//If Mario should walk to the castle (the flag has reached the bottom of the pole)
		{
			if (Timer::getLevelTimeLeft(dt) > 0)	//If we still have unbanked time to add to the score
			{
				if (Timer::setWaitTime(0.02, dt))	//Every 0.05s
				{
					Level::audioManager->playSoundByName("ScoreBeep");
					Timer::setLevelTimeLeft(Timer::getLevelTimeLeft(dt) - 1);	//We take a second off of the time
					timeLeftText.setString(std::to_string(Timer::getLevelTimeLeft(dt)));	//We then update the time string to show the "subtraction" effect
					mario.addScore(50);	//We add 50 score for this second left
				}
			}

			else    //All the time has been converted to score
			{
				if (Timer::setWaitTime(3, dt))	//After 3 seconds
				{
					window->setView(window->getDefaultView());
					currentLevel = QueryLevel(LevelType_YouWin);	//Go to the win screen
					reset();	//We call reset to set up the level for the next run when the player restarts
					mario.setLives(3);	//We set Mario's lives back to 3
					mario.removeCoins();						//	
					mario.removeScore(mario.getScore());		//		We remove all accumulated coins and score from Mario
					Level::audioManager->playMusicByName("Victory", false);
				}
			}
		}
	}

	else if (Timer::getLevelTimeLeft(dt) <= 0)	//Else if we've run out of time
	{
		mario.kill();	//We kill Mario
	}

	else if (mario.isAlive())	//Else if Mario is alive
	{
		if (!isPaused)	//If we're not paused
		{
			if (!hasBackgroundStarted)
			{
				audioManager->playMusicByName("LevelTheme", true);
				hasBackgroundStarted = true;
			}

			timeLeftText.setString(std::to_string(Timer::getLevelTimeLeft(dt)));	//We update the time left string

			if (mario.getPosition().x >= levelView.getCenter().x && mario.getVelocity().x > 0.f)	//If we've past by the centre of the level view, we need to scroll the view
			{
				levelView.reset(sf::FloatRect(mario.getPosition().x - 512, 0, 1024, 768));	//We set the view's new position to the centre of the screen relative to Mario

				//	//	//	//
				marioText.setPosition(mario.getPosition().x - 442, marioText.getPosition().y);
				scoreText.setPosition(mario.getPosition().x - 442, scoreText.getPosition().y);
				coinsText.setPosition(mario.getPosition().x - 132, coinsText.getPosition().y);
				coinBesideCount.setPosition(mario.getPosition().x - 172, coinBesideCount.getPosition().y);
				worldText.setPosition(mario.getPosition().x + 88, worldText.getPosition().y);		//	We update the positions of all our text so they also scroll with the screen
				levelText.setPosition(mario.getPosition().x + 113, levelText.getPosition().y);
				timeText.setPosition(mario.getPosition().x + 338, timeText.getPosition().y);
				timeLeftText.setPosition(mario.getPosition().x + 338, timeLeftText.getPosition().y);
				pausedText.setPosition(mario.getPosition().x, 384);
				//	//	//	//
			}

			if (mario.getPosition().x <= levelView.getCenter().x - 512)	//If Mario tries to walk back to the left
			{
				mario.setPosition(levelView.getCenter().x - 512, mario.getPosition().y);	//We stop him and set him back to the left most edge of the screen
			}
		}
	}

	//	//	//	//
	mario.update(dt);
	Level::enemyManager->update(dt, &mario, itemManager);
	Level::moveablesManager->update(dt, &mario, itemManager);		//Update Mario and all of our managers
	Level::itemManager->update(dt, &mario);
	//	//	//	//

	coinBesideCount.update(dt);	//We also update the coin beside the coin count so it glows
}

void MainGame::handleInput(float dt)
{
	if (userInput->isKeyDown(sf::Keyboard::Escape))	//Escape pressed
	{
		audioManager->playSoundByName("Pause");
		userInput->setKeyUp(sf::Keyboard::Escape);	//Set the key up
		if (isPaused)	//If we're paused
		{
			isPaused = false;	//We unpause
			mario.inputsAllowed = true;	//We can input to Mario again
			Timer::timerStopped = false; //Let the timer continue
			hasBackgroundStarted = false;
		}

		else
		{
			isPaused = true;	//Else, we pause the game
			mario.inputsAllowed = false;	//We ignore any Mario input while paused
			Timer::timerStopped = true;	//Stop the timer so we don't time the player when paused
			audioManager->stopAllMusic();
		}
	}

	mario.handleInput(dt);	//Handle Mario input, passing dt
}

void MainGame::render()
{
	window->setView(levelView);	//Set the viewable portion of the level to levelView
	
	window->draw(backgroundSprite);	//Draw the background

	//	//	//	//
	Level::itemManager->render(window);
	Level::enemyManager->render(window);		//	Render everything from our managers
	Level::moveablesManager->render(window);
	//	//	//	//

	//	//	//	//
	window->draw(marioText);
	window->draw(scoreText);
	window->draw(coinsText);
	window->draw(coinBesideCount);		//	Render all our text objects
	window->draw(worldText);
	window->draw(levelText);
	window->draw(timeText);
	window->draw(timeLeftText);
	//	//	//	//	

	window->draw(mario);	//Draw Mario

	if (isPaused)	//If we're paused
	{
		window->draw(pausedText);	//We also draw the paused game text
	}
}